/*
 * Decompiled with CFR 0.152.
 */
package io.konik.validation;

import com.google.common.base.Function;
import com.neovisionaries.i18n.CurrencyCode;
import io.konik.util.Amounts;
import io.konik.util.Items;
import io.konik.util.MonetarySummations;
import io.konik.zugferd.Invoice;
import io.konik.zugferd.entity.AllowanceCharge;
import io.konik.zugferd.entity.AppliedTax;
import io.konik.zugferd.entity.GrossPrice;
import io.konik.zugferd.entity.LogisticsServiceCharge;
import io.konik.zugferd.entity.SpecifiedAllowanceCharge;
import io.konik.zugferd.entity.Tax;
import io.konik.zugferd.entity.trade.MonetarySummation;
import io.konik.zugferd.entity.trade.Settlement;
import io.konik.zugferd.entity.trade.TradeTax;
import io.konik.zugferd.entity.trade.item.Item;
import io.konik.zugferd.entity.trade.item.ItemTax;
import io.konik.zugferd.entity.trade.item.SpecifiedAgreement;
import io.konik.zugferd.entity.trade.item.SpecifiedMonetarySummation;
import io.konik.zugferd.entity.trade.item.SpecifiedSettlement;
import io.konik.zugferd.unece.codes.TaxCategory;
import io.konik.zugferd.unece.codes.TaxCode;
import io.konik.zugferd.unqualified.Amount;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AmountCalculator {
    protected static Logger log = LoggerFactory.getLogger(AmountCalculator.class);

    public static RecalculationResult recalculate(Invoice invoice) {
        AmountCalculator.assertNotNull(invoice);
        CurrencyCode currency = AmountCalculator.getCurrency(invoice);
        List<Item> items = Items.purchasableItemsOnly(invoice.getTrade().getItems());
        Settlement settlement = invoice.getTrade().getSettlement();
        TaxAggregator taxAggregator = new TaxAggregator();
        if (items.isEmpty()) {
            return new RecalculationResult(MonetarySummations.newMonetarySummation(settlement.getMonetarySummation()), taxAggregator);
        }
        MonetarySummation monetarySummation = MonetarySummations.newMonetarySummation(currency);
        monetarySummation.setAllowanceTotal(new InvoiceAllowanceTotalCalculator().apply(settlement));
        monetarySummation.setChargeTotal(new InvoiceChargeTotalCalculator().apply(settlement));
        log.debug("Starting recalculating line total from {} items...", (Object)items.size());
        int itemsCounter = 0;
        for (Item item : items) {
            log.debug("==> {}:", (Object)(++itemsCounter));
            log.debug("Recalculating item: [{}]", (Object)(item.getProduct() != null ? item.getProduct().getName() : "N/A"));
            Amount lineTotal = new ItemLineTotalCalculator().apply(item);
            ItemTax itemTax = new ItemTaxExtractor().apply(item);
            log.debug("Recalculated item line total = {}", (Object)lineTotal);
            log.debug("Recalculated item tax = {}%", (Object)itemTax.getPercentage());
            taxAggregator.add(itemTax, lineTotal != null ? lineTotal.getValue() : BigDecimal.ZERO);
            monetarySummation.setLineTotal(Amounts.add(monetarySummation.getLineTotal(), lineTotal));
            log.debug("Current monetarySummation.lineTotal = {} (the sum of all line totals)", (Object)monetarySummation.getLineTotal());
        }
        log.debug("==> DONE!");
        log.debug("Finished recalculating monetarySummation.lineTotal...");
        AmountCalculator.appendTaxFromInvoiceAllowanceCharge(settlement, taxAggregator);
        AmountCalculator.appendTaxFromInvoiceServiceCharge(settlement, taxAggregator);
        monetarySummation.setTaxBasisTotal(new Amount(taxAggregator.calculateTaxBasis(), currency));
        monetarySummation.setTaxTotal(new Amount(taxAggregator.calculateTaxTotal(), currency));
        monetarySummation.setGrandTotal(Amounts.add(monetarySummation.getTaxBasisTotal(), monetarySummation.getTaxTotal()));
        log.debug("Recalculated grand total = {} (tax basis total + tax total)", (Object)monetarySummation.getGrandTotal());
        if (settlement.getMonetarySummation() != null && settlement.getMonetarySummation().getTotalPrepaid() != null) {
            monetarySummation.setTotalPrepaid(settlement.getMonetarySummation().getTotalPrepaid());
        }
        monetarySummation.setDuePayable(Amounts.add(monetarySummation.getGrandTotal(), Amounts.negate(monetarySummation.getTotalPrepaid())));
        MonetarySummation result = MonetarySummations.precise(monetarySummation, 2, RoundingMode.HALF_UP);
        log.debug("Recalculating invoice monetary summation DONE!");
        log.debug(" ==> result: {}", (Object)result);
        log.debug("");
        return new RecalculationResult(result, taxAggregator);
    }

    public static SpecifiedMonetarySummation calculateSpecifiedMonetarySummation(Item item) {
        log.debug("Recalculating specified monetary summation for [{}]", (Object)(item.getProduct() != null ? item.getProduct().getName() : "N/A"));
        CurrencyCode currencyCode = AmountCalculator.getCurrency(item);
        SpecifiedMonetarySummation monetarySummation = MonetarySummations.newSpecifiedMonetarySummation(currencyCode);
        monetarySummation.setLineTotal(Amounts.setPrecision(new ItemLineTotalCalculator().apply(item), 2, RoundingMode.HALF_UP));
        monetarySummation.setTotalAllowanceCharge(Amounts.setPrecision(new ItemTotalAllowanceChargeCalculator(currencyCode).apply(item), 2, RoundingMode.HALF_UP));
        log.debug("==> lineTotal = {}", (Object)monetarySummation.getLineTotal());
        log.debug("==> totalAllowanceCharge = {}", (Object)monetarySummation.getTotalAllowanceCharge());
        return monetarySummation;
    }

    private static void appendTaxFromInvoiceServiceCharge(Settlement settlement, TaxAggregator taxAggregator) {
        log.debug("Adding tax amounts from invoice service charge...");
        if (settlement.getServiceCharge() != null) {
            for (LogisticsServiceCharge charge : settlement.getServiceCharge()) {
                if (charge.getTradeTax() == null || charge.getAmount() == null) continue;
                for (AppliedTax tax : charge.getTradeTax()) {
                    log.debug("==> added {} to {}%", (Object)charge.getAmount(), (Object)tax.getPercentage());
                    taxAggregator.add(tax, charge.getAmount().getValue());
                }
            }
        }
    }

    private static void appendTaxFromInvoiceAllowanceCharge(Settlement settlement, TaxAggregator taxAggregator) {
        log.debug("Adding tax amounts from invoice allowance charge...");
        if (settlement.getAllowanceCharge() != null) {
            for (SpecifiedAllowanceCharge charge : settlement.getAllowanceCharge()) {
                if (charge.getCategory() == null || charge.getActual() == null) continue;
                BigDecimal amount = charge.getActual().getValue();
                if (charge.isDiscount()) {
                    amount = amount.negate();
                }
                log.debug("==> added {} to {}%", (Object)amount, (Object)charge.getCategory().getPercentage());
                taxAggregator.add(charge.getCategory(), amount);
            }
        }
    }

    public static CurrencyCode getCurrency(Invoice invoice) {
        AmountCalculator.assertNotNull(invoice);
        return invoice.getTrade().getSettlement().getCurrency();
    }

    public static CurrencyCode getCurrency(Item item) {
        AmountCalculator.assertNotNull(item);
        SpecifiedAgreement agreement = item.getAgreement();
        if (agreement != null && agreement.getGrossPrice() != null && agreement.getGrossPrice().getChargeAmount() != null) {
            return agreement.getGrossPrice().getChargeAmount().getCurrency();
        }
        if (agreement != null && agreement.getNetPrice() != null && agreement.getNetPrice().getChargeAmount() != null) {
            return agreement.getNetPrice().getChargeAmount().getCurrency();
        }
        SpecifiedSettlement settlement = item.getSettlement();
        if (settlement != null && settlement.getMonetarySummation() != null && settlement.getMonetarySummation().getLineTotal() != null) {
            return settlement.getMonetarySummation().getLineTotal().getCurrency();
        }
        return null;
    }

    private static void assertNotNull(Invoice invoice) {
        if (invoice == null || invoice.getTrade() == null) {
            throw new IllegalArgumentException("Invoice and Trade objects cannot be null");
        }
    }

    private static void assertNotNull(Item item) {
        if (item == null) {
            throw new IllegalArgumentException("Item cannot be null");
        }
    }

    public static final class RecalculationResult {
        private final MonetarySummation monetarySummation;
        private final TaxAggregator taxAggregator;

        public RecalculationResult(MonetarySummation monetarySummation, TaxAggregator taxAggregator) {
            this.monetarySummation = monetarySummation;
            this.taxAggregator = taxAggregator;
        }

        public MonetarySummation getMonetarySummation() {
            return this.monetarySummation;
        }

        public TaxAggregator getTaxAggregator() {
            return this.taxAggregator;
        }
    }

    public static final class TaxAggregator {
        private static final int PRECISION = 2;
        private static final RoundingMode ROUNDING_MODE = RoundingMode.HALF_UP;
        private final ConcurrentMap<Key, BigDecimal> map = new ConcurrentHashMap<Key, BigDecimal>();

        public void add(Tax tax, BigDecimal amount) {
            Key key = Key.create(tax);
            this.map.putIfAbsent(key, BigDecimal.ZERO);
            this.map.put(key, ((BigDecimal)this.map.get(key)).add(amount));
        }

        public BigDecimal getTaxBasisForTaxPercentage(BigDecimal percentage) {
            BigDecimal value = BigDecimal.ZERO;
            for (Key key : this.map.keySet()) {
                if (!percentage.equals(key.getPercentage())) continue;
                value = value.add((BigDecimal)this.map.get(key));
            }
            return value;
        }

        public BigDecimal calculateTaxBasis() {
            log.debug("Recalculating tax basis for tax percentages: {}", (Object)Arrays.toString(this.map.keySet().toArray()));
            BigDecimal taxBasis = BigDecimal.ZERO;
            for (BigDecimal amount : this.map.values()) {
                taxBasis = taxBasis.add(amount);
            }
            log.debug("Recalculated tax basis = {}", (Object)taxBasis);
            return taxBasis;
        }

        public BigDecimal calculateTaxTotal() {
            log.debug("Calculating tax total...");
            BigDecimal taxTotal = BigDecimal.ZERO;
            for (Map.Entry entry : this.map.entrySet()) {
                BigDecimal percentage = ((Key)entry.getKey()).getPercentage();
                BigDecimal value = (BigDecimal)entry.getValue();
                BigDecimal taxAmount = TaxAggregator.calculateTaxAmount(percentage, value);
                log.debug("===> {} x {}% = {}", new Object[]{value, percentage, taxAmount});
                taxTotal = taxTotal.add(taxAmount);
            }
            log.debug("Recalculated tax total = {}", (Object)taxTotal);
            return taxTotal;
        }

        public List<TradeTax> generateTradeTaxList(CurrencyCode currencyCode, List<TradeTax> previousList) {
            LinkedList<TradeTax> taxes = new LinkedList<TradeTax>();
            for (Key key : this.map.keySet()) {
                TradeTax tradeTax = new TradeTax();
                tradeTax.setType(key.getCode());
                tradeTax.setCategory(key.getCategory());
                tradeTax.setPercentage(key.getPercentage());
                BigDecimal basis = (BigDecimal)this.map.get(key);
                BigDecimal calculated = TaxAggregator.calculateTaxAmount(key.getPercentage(), basis);
                tradeTax.setBasis(new Amount(basis, currencyCode));
                tradeTax.setCalculated(new Amount(calculated, currencyCode));
                TradeTax existing = null;
                if (previousList != null) {
                    for (TradeTax current : previousList) {
                        if (!tradeTax.getType().equals((Object)current.getType()) || !tradeTax.getCategory().equals((Object)current.getCategory()) || !tradeTax.getPercentage().equals(current.getPercentage())) continue;
                        existing = current;
                        break;
                    }
                }
                if (existing != null) {
                    tradeTax.setExemptionReason(existing.getExemptionReason());
                    if (existing.getAllowanceCharge() != null) {
                        tradeTax.setAllowanceCharge(new Amount(existing.getAllowanceCharge().getValue(), existing.getAllowanceCharge().getCurrency()));
                    }
                    if (existing.getLineTotal() != null) {
                        tradeTax.setLineTotal(new Amount(existing.getLineTotal().getValue(), existing.getLineTotal().getCurrency()));
                    }
                }
                taxes.add(tradeTax);
            }
            return taxes;
        }

        public static BigDecimal calculateTaxAmount(BigDecimal percentage, BigDecimal value) {
            return value.multiply(percentage.divide(BigDecimal.valueOf(100L))).setScale(2, ROUNDING_MODE);
        }

        public String toString() {
            return "TaxAggregator{map=" + this.map + '}';
        }

        static final class Key {
            private final BigDecimal percentage;
            private final TaxCode code;
            private final TaxCategory category;

            private Key(Tax tax) {
                this.percentage = tax.getPercentage();
                this.category = tax.getCategory();
                this.code = tax.getType();
            }

            public static Key create(Tax tax) {
                return new Key(tax);
            }

            public BigDecimal getPercentage() {
                return this.percentage;
            }

            public TaxCode getCode() {
                return this.code;
            }

            public TaxCategory getCategory() {
                return this.category;
            }

            public boolean equals(Object o) {
                if (this == o) {
                    return true;
                }
                if (!(o instanceof Key)) {
                    return false;
                }
                Key key = (Key)o;
                if (!this.percentage.equals(key.percentage)) {
                    return false;
                }
                if (this.code != key.code) {
                    return false;
                }
                return this.category == key.category;
            }

            public int hashCode() {
                int result = this.percentage.hashCode();
                result = 31 * result + (this.code != null ? this.code.hashCode() : 0);
                result = 31 * result + (this.category != null ? this.category.hashCode() : 0);
                return result;
            }

            public String toString() {
                return "Key{percentage=" + this.percentage + ", code=" + (Object)((Object)this.code) + ", category=" + (Object)((Object)this.category) + '}';
            }
        }
    }

    public static final class ItemTaxExtractor
    implements Function<Item, ItemTax> {
        @Nullable
        public ItemTax apply(@Nullable Item item) {
            if (item == null || item.getSettlement() == null) {
                return null;
            }
            if (item.getSettlement().getTradeTax().isEmpty()) {
                return null;
            }
            return item.getSettlement().getTradeTax().get(0);
        }
    }

    public static final class InvoiceChargeTotalCalculator
    implements Function<Settlement, Amount> {
        @Nullable
        public Amount apply(@Nullable Settlement settlement) {
            if (settlement == null || settlement.getAllowanceCharge() == null) {
                return null;
            }
            BigDecimal chargeValue = BigDecimal.ZERO;
            for (SpecifiedAllowanceCharge specifiedAllowanceCharge : settlement.getAllowanceCharge()) {
                if (!specifiedAllowanceCharge.isSurcharge()) continue;
                chargeValue = chargeValue.add(specifiedAllowanceCharge.getActual().getValue());
            }
            for (LogisticsServiceCharge logisticsServiceCharge : settlement.getServiceCharge()) {
                chargeValue = chargeValue.add(logisticsServiceCharge.getAmount().getValue());
            }
            Amount amount = new Amount(chargeValue, settlement.getCurrency());
            log.debug("Invoice charge total = {}", (Object)amount);
            return amount;
        }
    }

    public static final class InvoiceAllowanceTotalCalculator
    implements Function<Settlement, Amount> {
        @Nullable
        public Amount apply(@Nullable Settlement settlement) {
            if (settlement == null || settlement.getAllowanceCharge() == null) {
                return null;
            }
            BigDecimal chargeValue = BigDecimal.ZERO;
            for (SpecifiedAllowanceCharge charge : settlement.getAllowanceCharge()) {
                if (!charge.isDiscount()) continue;
                chargeValue = chargeValue.add(charge.getActual().getValue());
            }
            Amount amount = new Amount(chargeValue, settlement.getCurrency());
            log.debug("Invoice allowance total = {}", (Object)amount);
            return amount;
        }
    }

    public static final class ItemTotalAllowanceChargeCalculator
    implements Function<Item, Amount> {
        private final CurrencyCode currencyCode;

        public ItemTotalAllowanceChargeCalculator(CurrencyCode currencyCode) {
            this.currencyCode = currencyCode;
        }

        public Amount apply(@Nullable Item item) {
            GrossPrice grossPrice;
            Amount totalAllowanceCharge = Amounts.zero(this.currencyCode);
            BigDecimal quantity = BigDecimal.ONE;
            if (item != null && item.getDelivery() != null && item.getDelivery().getBilled() != null) {
                quantity = item.getDelivery().getBilled().getValue();
            }
            if (item != null && item.getAgreement() != null && item.getAgreement().getGrossPrice() != null && (grossPrice = item.getAgreement().getGrossPrice()).getAllowanceCharges() != null && !grossPrice.getAllowanceCharges().isEmpty()) {
                for (AllowanceCharge charge : grossPrice.getAllowanceCharges()) {
                    BigDecimal chargeValue = charge.getActual().getValue();
                    if (charge.isDiscount()) {
                        chargeValue = chargeValue.negate();
                    }
                    Amount amount = new Amount(chargeValue.multiply(quantity), this.currencyCode);
                    totalAllowanceCharge = Amounts.add(totalAllowanceCharge, amount);
                }
                totalAllowanceCharge = Amounts.setPrecision(totalAllowanceCharge, 2, RoundingMode.HALF_UP);
            }
            return totalAllowanceCharge;
        }
    }

    static final class ItemTaxTotalCalculator
    implements Function<Item, Amount> {
        private final Amount lineTotal;

        public ItemTaxTotalCalculator(Amount lineTotal) {
            this.lineTotal = lineTotal;
        }

        public Amount apply(@Nullable Item item) {
            CurrencyCode currency = this.lineTotal.getCurrency();
            Amount taxTotal = Amounts.zero(currency);
            if (item != null && item.getSettlement() != null && item.getSettlement().getTradeTax() != null) {
                for (ItemTax tax : item.getSettlement().getTradeTax()) {
                    BigDecimal taxRate = tax.getPercentage().divide(BigDecimal.valueOf(100L), 2, 4).setScale(2, RoundingMode.HALF_UP);
                    BigDecimal taxValue = this.lineTotal.getValue().multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
                    taxTotal = Amounts.add(taxTotal, new Amount(taxValue, currency));
                }
            }
            return taxTotal;
        }
    }

    static final class ItemLineTotalCalculator
    implements Function<Item, Amount> {
        ItemLineTotalCalculator() {
        }

        @Nullable
        public Amount apply(@Nullable Item item) {
            Amount originLineTotal = null;
            if (item != null && item.getSettlement() != null && item.getSettlement().getMonetarySummation() != null) {
                originLineTotal = Amounts.copy(item.getSettlement().getMonetarySummation().getLineTotal());
            }
            if (item == null || item.getDelivery() == null || item.getAgreement() == null) {
                return originLineTotal;
            }
            if (item.getAgreement().getNetPrice() == null) {
                return originLineTotal;
            }
            BigDecimal quantity = item.getDelivery().getBilled() != null ? item.getDelivery().getBilled().getValue() : BigDecimal.ZERO;
            BigDecimal basisQuantity = Items.basisQuantity(item);
            Amount amount = item.getAgreement().getNetPrice().getChargeAmount();
            log.debug("Line total formula: {} (net price) / {} (BasisQuantity)  x {} (quantity)", new Object[]{amount, basisQuantity, quantity});
            Amount netPricePerUnit = Amounts.divide(amount, basisQuantity);
            return Amounts.multiply(netPricePerUnit, quantity);
        }
    }
}

